﻿Imports Microsoft.VisualBasic
Imports System.CodeDom
Imports System.CodeDom.Compiler
Imports System.Reflection

' This class is responsible for generating source code for 
' classes that wrap interfaces or other classes in a way that 
' is suitble for coding a proxy or decorator class.
Public Class WrapperGenerator
    ' A model of the code to be generated.
    Private codeModel As CodeNamespace

    ' The name of the generated class's private instance variable
    Private Const VAR_NAME As String = "_myWrappedObject_"

    ' The name of the generated class's protected readonly property for 
    ' accessing the wrapped object
    Private Const PROP_NAME As String = "_WrappedObject_"

    ' Constructor
    ' The argument for this constructor is the Type of the 
    ' class or interface that this object will generate a 
    ' wrapper for.
    Public Sub New(ByVal base As Type)
        codeModel = GenerateCodeModel(base)
    End Sub

    Public Sub GenerateCode(ByVal generator As ICodeGenerator)
        Dim options As CodeGeneratorOptions = New CodeGeneratorOptions()
        generator.GenerateCodeFromNamespace(codeModel, Console.Out, options)
    End Sub

    ' Return a CodeCompileUnit that is a model for a wrapper
    ' of the given class or interface.
    ' The parameter is a Type object that must represent a
    ' class or an interface.
    Private Function GenerateCodeModel(ByVal base As Type) _
                    As CodeNamespace
        If Not (base.IsClass OrElse base.IsInterface) Then
            Dim msg As String
            msg = "Argument must be a Type object that" _
                + " represents a class or interface."
            Throw New ArgumentException(msg, "base")
        End If

        Dim wrapper As CodeTypeDeclaration
        wrapper = New CodeTypeDeclaration("Wrapper" + base.Name)
        GenerateBaseTypes(wrapper, base)
        wrapper.TypeAttributes = TypeAttributes.Public
        GenerateMembers(wrapper.Members, base)
        wrapper.IsClass = True

        Dim model As CodeNamespace
        model = New CodeNamespace("Wrappers")
        model.Types.Add(wrapper)
        Return model
    End Function

    ' Generate members for the wrapper class
    Private Sub GenerateMembers(ByVal members As CodeTypeMemberCollection, _
                                ByVal base As Type)
        members.Add(GenerateVariableMember(base))
        members.Add(GenerateProperty(base))
        members.Add(GenerateConstructor(base))
        Dim memberInfoArray As MemberInfo() = base.GetMembers()
        Dim info As MemberInfo
        For Each info In memberInfoArray
            Dim thisMember As CodeTypeMember = GenerateMember(info)
            If Not (thisMember Is Nothing) Then
                members.Add(thisMember)
            End If
        Next info
    End Sub

    ' Generate a member in the model
    Private Function GenerateMember(ByVal info As MemberInfo) As CodeTypeMember
        If TypeOf info Is PropertyInfo Then
            Return GeneratePropertyMember(DirectCast(info, PropertyInfo))
        ElseIf TypeOf info Is MethodInfo Then
            Return GenerateMethodMember(DirectCast(info, MethodInfo))
        End If
        Return Nothing
    End Function

    ' Generate a method in the code model
    Private Function GenerateMethodMember(ByVal info As MethodInfo) _
                    As CodeMemberMethod
        If info.Name.StartsWith("set_") Or info.Name.StartsWith("get_") Then
            Return Nothing
        End If
        Dim method As CodeMemberMethod = New CodeMemberMethod()
        method.Name = info.Name

        SetCodeTypeMemberAttribute(method, MemberAttributes.Abstract, _
                                   info.IsAbstract)
        SetCodeTypeMemberAttribute(method, MemberAttributes.Assembly, _
                                   info.IsAssembly)
        SetCodeTypeMemberAttribute(method, MemberAttributes.Family, _
                                   info.IsFamily)
        SetCodeTypeMemberAttribute(method, MemberAttributes.FamilyAndAssembly, _
                                   info.IsFamilyAndAssembly)
        SetCodeTypeMemberAttribute(method, MemberAttributes.FamilyOrAssembly, _
                                   info.IsFamilyOrAssembly)
        SetCodeTypeMemberAttribute(method, MemberAttributes.Final, _
                                   info.IsFinal)
        SetCodeTypeMemberAttribute(method, MemberAttributes.Private, _
                                   info.IsPrivate)
        SetCodeTypeMemberAttribute(method, MemberAttributes.Public, _
                                   info.IsPublic)
        SetCodeTypeMemberAttribute(method, MemberAttributes.Static, _
                                   info.IsStatic)

        Dim declaringType As Type = info.DeclaringType
        If declaringType.IsInterface Then
            method.ImplementationTypes.Add(New CodeTypeReference(declaringType))
        End If
        GenerateParameters(method.Parameters, info.GetParameters())
        method.ReturnType = New CodeTypeReference(info.ReturnType)

        Dim invokeExpr As CodeMethodInvokeExpression
        invokeExpr = GenerateMethodInvocation(method.Name, method.Parameters)

        If info.ReturnType.FullName = "System.Void" Then
            method.Statements.Add(New CodeExpressionStatement(invokeExpr))
        Else
            method.Statements.Add(New CodeMethodReturnStatement(invokeExpr))
        End If

        Return method
    End Function

    ' The the specified attribute of the specified member to true or false
    Private Sub SetCodeTypeMemberAttribute(ByVal member As CodeTypeMember, _
                                           ByVal attr As MemberAttributes, _
                                           ByVal value As Boolean)
        If value Then
            member.Attributes = member.Attributes Or attr
        Else
            member.Attributes = member.Attributes And Not attr
        End If
    End Sub

    ' Generate a method invocation to the wrapped object
    Private Function GenerateMethodInvocation(ByVal methodName As String, _
                                              ByVal parameters As CodeParameterDeclarationExpressionCollection) _
                      As CodeMethodInvokeExpression
        Dim var As CodeVariableReferenceExpression
        var = New CodeVariableReferenceExpression(VAR_NAME)

        Dim method As CodeMethodReferenceExpression
        method = New CodeMethodReferenceExpression(var, methodName)

        Dim invocation As CodeMethodInvokeExpression
        invocation = New CodeMethodInvokeExpression()
        invocation.Method = method

        Dim parameter As CodeParameterDeclarationExpression
        For Each parameter In parameters
            Dim thisParam As CodeVariableReferenceExpression
            thisParam = New CodeVariableReferenceExpression(parameter.Name)
            invocation.Parameters.Add(thisParam)
        Next

        Return invocation
    End Function

    ' Generate a property member in the model
    Private Function GeneratePropertyMember(ByVal info As PropertyInfo) _
                    As CodeMemberProperty
        Dim prop As CodeMemberProperty = New CodeMemberProperty()
        prop.Name = info.Name
        prop.Type = New CodeTypeReference(info.PropertyType)
        prop.HasGet = info.CanRead
        prop.HasSet = info.CanWrite

        prop.Attributes = prop.Attributes Or MemberAttributes.Public
        prop.Attributes = prop.Attributes Or MemberAttributes.Override

        GenerateParameters(prop.Parameters, info.GetIndexParameters())
        Dim declaringType As Type = info.DeclaringType
        If declaringType.IsInterface Then
            prop.ImplementationTypes.Add(New CodeTypeReference(declaringType))
        End If

        Dim var As CodeVariableReferenceExpression
        var = New CodeVariableReferenceExpression(VAR_NAME)
        Dim propRef As CodePropertyReferenceExpression
        propRef = New CodePropertyReferenceExpression(var, prop.Name)

        If prop.HasGet Then
            prop.GetStatements.Add(New CodeMethodReturnStatement(propRef))
        End If
        If prop.HasSet Then
            Dim newValue As CodePropertySetValueReferenceExpression
            newValue = New CodePropertySetValueReferenceExpression()
            prop.SetStatements.Add(New CodeAssignStatement(propRef, newValue))
        End If

        Return prop
    End Function

    Private Sub GenerateParameters(ByVal parameters As CodeParameterDeclarationExpressionCollection, _
                                   ByVal parameterArray As ParameterInfo())
        Dim thisParameter As ParameterInfo
        For Each thisParameter In parameterArray
            Dim paramType As Type = thisParameter.ParameterType
            Dim name As String = thisParameter.Name
            Dim decl As CodeParameterDeclarationExpression
            decl = New CodeParameterDeclarationExpression(paramType, name)
            If thisParameter.IsIn Then
                If thisParameter.IsOut Then
                    decl.Direction = FieldDirection.Ref
                Else
                    decl.Direction = FieldDirection.In
                End If
            Else
                decl.Direction = FieldDirection.Out
            End If
            parameters.Add(decl)
        Next
    End Sub

    ' Generate a constructor fo the wrapper class
    Private Function GenerateConstructor(ByVal base As Type) As CodeConstructor
        Dim constructor As CodeConstructor = New CodeConstructor()
        Dim parameterName As String = "my" + base.Name
        Dim parameter As CodeParameterDeclarationExpression
        parameter = New CodeParameterDeclarationExpression(base, parameterName)
        constructor.Parameters.Add(parameter)

        Dim statement As CodeAssignStatement = New CodeAssignStatement()
        statement.Left = New CodeVariableReferenceExpression(VAR_NAME)
        statement.Right = New CodeArgumentReferenceExpression(parameterName)
        constructor.Statements.Add(statement)
        Return constructor
    End Function

    ' Generate the wrapper class's private instance variable
    Private Function GenerateVariableMember(ByVal base As Type) _
                    As CodeMemberField
        Dim variable As CodeMemberField
        variable = New CodeMemberField(base, VAR_NAME)
        variable.Attributes = MemberAttributes.Private
        Dim varCommentString As String = "Reference to wrapped object"
        variable.Comments.Add(New CodeCommentStatement(varCommentString))
        Return variable
    End Function

    Private Function GenerateProperty(ByVal base As Type) As CodeMemberProperty
        Dim prop As CodeMemberProperty
        prop = New CodeMemberProperty()
        prop.Name = PROP_NAME
        prop.Type = New CodeTypeReference(base)
        prop.HasGet = True
        prop.HasSet = False

        prop.Attributes = prop.Attributes Or MemberAttributes.Public

        Dim var As CodeVariableReferenceExpression
        var = New CodeVariableReferenceExpression(VAR_NAME)
        prop.GetStatements.Add(New CodeMethodReturnStatement(var))

        Return prop
    End Function

    Private Sub GenerateBaseTypes(ByVal wrapper As CodeTypeDeclaration, _
                                  ByVal base As Type)
        If base.IsInterface Then
            wrapper.BaseTypes.Add(base)
        Else
            Dim hasBaseType As Boolean
            hasBaseType = Not (base.BaseType Is Nothing Or base.BaseType Is GetType(Object))
            If hasBaseType Then
                wrapper.BaseTypes.Add(base.BaseType)
            End If

            Dim interfac As Type
            Dim interfaces As Type() = base.GetInterfaces()
            For Each interfac In interfaces
                wrapper.BaseTypes.Add(interfac)
            Next interfac
        End If
    End Sub
End Class
